#
# This file is part of LiteX-Boards.
#
# Copyright (c) 2023 Hans Baier <hansfbaier@gmail.com>
# SPDX-License-Identifier: BSD-2-Clause

from litex.build.generic_platform import Subsignal, Pins, IOStandard, Misc

class EnclustraST1:
    """
        the ST1 baseboard contains standard peripherals
        and can be used with a number of different FPGA core boards
        the default VCCIO values are from the jumper positions from the example
        design in the manual
    """
    def __init__(self, vccio_a="1.8V", vccio_b="1.8V", vccio_c="1.8V") -> None:
        def strip_v(s, is_a = False):
            assert s in [ 
                "1.2V" if not is_a else None, 
                "1.8V", # VCC_OUT_B
                "2.5V", # VCC_OUT_A
                "3.3V" ]
            return s.replace(".", "").replace("V", "")

        vccio_a = strip_v(vccio_a) # Bank 12, 14
        vccio_b = strip_v(vccio_b) # Bank 15, 16
        vccio_c = strip_v(vccio_c) # Bank 13

        self.io = [
            ("clk_ref", 0,
                Subsignal("p", Pins("C:7"), IOStandard("LVDS")),
                Subsignal("n", Pins("C:9"), IOStandard("LVDS"))
            ),
            ("clk_ref0", 0,
                Subsignal("p", Pins("B:3"), IOStandard("LVDS")),
                Subsignal("n", Pins("B:5"), IOStandard("LVDS"))
            ),
            ("clk_ref1", 0,
                Subsignal("p", Pins("C:10"), IOStandard("LVDS")),
                Subsignal("n", Pins("C:12"), IOStandard("LVDS"))
            ),
            ("clk_ref2", 0,
                Subsignal("p", Pins("C:3"), IOStandard("LVDS")),
                Subsignal("n", Pins("C:5"), IOStandard("LVDS"))
            ),
            # daughterboard LEDs
            ("user_led", 4, Pins("C:142"), IOStandard("LVCMOS" + vccio_c), Misc("SLEW=SLOW")),
            ("user_led", 5, Pins("C:144"), IOStandard("LVCMOS" + vccio_c), Misc("SLEW=SLOW")),
            ("i2c", 0,
                Subsignal("scl",     Pins("A:111"), Misc("SLEW=SLOW")),
                Subsignal("sda",     Pins("A:113"), Misc("SLEW=SLOW")),
                Subsignal("int_n",   Pins("A:115"), Misc("SLEW=SLOW")),
                IOStandard("LVCMOS" + vccio_a)
            ),
            ("i2c_fpga", 0,
                Subsignal("scl",     Pins("A:55"), Misc("SLEW=SLOW")),
                Subsignal("sda",     Pins("A:57"), Misc("SLEW=SLOW")),
                IOStandard("LVCMOS" + vccio_a)
            ),
            ("i2c_mipi_sel", 0, Pins("A:63"), IOStandard("LVCMOS" + vccio_a)),

            # HDMI Bus
            ("hdmi", 0,
                Subsignal("clk_p",   Pins("C:139")),
                Subsignal("clk_n",   Pins("C:141")),
                Subsignal("data0_p", Pins("C:45")),
                Subsignal("data0_n", Pins("C:47")),
                Subsignal("data1_p", Pins("C:51")),
                Subsignal("data1_n", Pins("C:53")),
                Subsignal("data2_p", Pins("C:57")),
                Subsignal("data2_n", Pins("C:59")),
                Subsignal("hpd",     Pins("A:61")),
                IOStandard("TMDS_33")
            ),
            ("sfp_tx", 0,  # SFP A
                Subsignal("p", Pins("C:63")),
                Subsignal("n", Pins("C:65"))
            ),
            ("sfp_rx", 0,  # SFP A
                Subsignal("p", Pins("C:66")),
                Subsignal("n", Pins("C:68"))
            ),
            ("usb3", 0,
                Subsignal("tx_p", Pins("B:29")),
                Subsignal("tx_n", Pins("B:33")),
                Subsignal("rx_p", Pins("B:32")),
                Subsignal("rx_n", Pins("B:36")),
            ),
            ("usb3", 1,
                Subsignal("tx_p", Pins("B:37")),
                Subsignal("tx_n", Pins("B:41")),
                Subsignal("rx_p", Pins("B:40")),
                Subsignal("rx_n", Pins("B:44")),
            ),
            ("displayport", 0,
                Subsignal("aux_in",  Pins("A:88")),
                Subsignal("aux_out", Pins("A:90")),
                Subsignal("aux_oe",  Pins("A:92")),
                Subsignal("hpd",     Pins("A:94")),
                Subsignal("lanes_p", Pins("B:21 B:13 B:16 B:24")),
                Subsignal("lanes_n", Pins("B:25 B:17 B:20 B:28")),
            ),
            ("mipi", 0,
                Subsignal("clkp", Pins("C:151")),
                Subsignal("clkn", Pins("C:153")),
                Subsignal("clk_d0lp_p", Pins("C:160")),
                Subsignal("clk_d0lp_n", Pins("C:162")),
                Subsignal("dp",   Pins("C:145 C:154")),
                Subsignal("dn",   Pins("C:147 C:156")),
                IOStandard("MIPI_DPHY")
            ),
            ("mipi", 1,
                Subsignal("clkp", Pins("C:148")),
                Subsignal("clkn", Pins("C:150")),
                Subsignal("clk_d0lp_p", Pins("C:161")),
                Subsignal("clk_d0lp_n", Pins("C:163")),
                Subsignal("dp",   Pins("C:164 C:157")),
                Subsignal("dn",   Pins("C:166 C:159")),
                IOStandard("MIPI_DPHY")
            ),
        ]

    connectors = [
    ("FMC", {
        "DP0_C2M_P":   "B:45",
        "DP0_C2M_N":   "B:47",
        "DP1_C2M_P":   "B:51",
        "DP1_C2M_N":   "B:53",
        "CLK1_M2C_P":  "B:57",
        "CLK1_M2C_N":  "B:59",
        "DP2_C2M_P":   "B:63",
        "DP2_C2M_N":   "B:65",
        "DP3_C2M_P":   "B:69",
        "DP3_C2M_N":   "B:71",
        "LA33_P":      "B:75",
        "LA33_N":      "B:77",
        "LA32_P":      "B:81",
        "LA32_N":      "B:83",
        "LA31_P":      "B:87",
        "LA31_N":      "B:89",
        "LA30_P":      "B:91",
        "LA30_N":      "B:93",
        "LA29_P":      "B:97",
        "LA29_N":      "B:99",
        "LA28_P":      "B:101",
        "LA28_N":      "B:103",
        "LA27_P":      "B:107",
        "LA27_N":      "B:109",
        "LA26_P":      "B:111",
        "LA26_N":      "B:113",
        "LA25_P":      "B:117",
        "LA25_N":      "B:119",
        "LA18_CC_P":   "B:123",
        "LA18_CC_N":   "B:125",
        "LA24_P":      "B:129",
        "LA24_N":      "B:131",
        "LA23_P":      "B:133",
        "LA23_N":      "B:135",
        "LA22_P":      "B:139",
        "LA22_N":      "B:141",
        "LA21_P":      "B:145",
        "LA21_N":      "B:147",
        "LA17_CC_P":   "B:151",
        "LA17_CC_N":   "B:153",
        "LA20_P":      "B:157",
        "LA20_N":      "B:159",
        "LA19_P":      "B:163",
        "LA19_N":      "B:165",
        "DP0_M2C_P":   "B:48",
        "DP0_M2C_N":   "B:50",
        "DP1_M2C_P":   "B:54",
        "DP1_M2C_N":   "B:56",
        "DP2_M2C_P":   "B:60",
        "DP2_M2C_N":   "B:62",
        "DP3_M2C_P":   "B:66",
        "DP3_M2C_N":   "B:68",
        "LA16_P":      "B:72",
        "LA16_N":      "B:74",
        "CLK0_M2C_P":  "B:78",
        "CLK0_M2C_N":  "B:80",
        "LA15_P":      "B:84",
        "LA15_N":      "B:86",
        "LA14_P":      "B:90",
        "LA14_N":      "B:92",
        "LA13_P":      "B:94",
        "LA13_N":      "B:96",
        "LA12_P":      "B:100",
        "LA12_N":      "B:102",
        "LA11_P":      "B:104",
        "LA11_N":      "B:106",
        "LA10_P":      "B:110",
        "LA10_N":      "B:112",
        "LA09_P":      "B:114",
        "LA09_N":      "B:116",
        "LA08_P":      "B:120",
        "LA08_N":      "B:122",
        "LA07_P":      "B:124",
        "LA07_N":      "B:126",
        "LA01_CC_P":   "B:130",
        "LA01_CC_N":   "B:132",
        "LA06_P":      "B:136",
        "LA06_N":      "B:138",
        "LA05_P":      "B:142",
        "LA05_N":      "B:144",
        "LA04_P":      "B:148",
        "LA04_N":      "B:150",
        "LA00_CC_P":   "B:154",
        "LA00_CC_N":   "B:156",
        "LA03_P":      "B:160",
        "LA03_N":      "B:162",
        "LA02_P":      "B:164",
        "LA02_N":      "B:166",

        "DP4_C2M_P":   "C:13",
        "DP4_C2M_N":   "C:17",
        "DP5_C2M_P":   "C:21",
        "DP5_C2M_N":   "C:25",
        "DP6_C2M_P":   "C:29",
        "DP6_C2M_N":   "C:33",
        "DP7_C2M_P":   "C:37",
        "DP7_C2M_N":   "C:41",
        "HA13_P":      "C:69",
        "HA13_N":      "C:71",
        "HA11_P":      "C:75",
        "HA11_N":      "C:77",
        "HA09_P":      "C:79",
        "HA09_N":      "C:81",
        "HA07_P":      "C:85",
        "HA07_N":      "C:87",
        "HA01_CC_P":   "C:89",
        "HA01_CC_N":   "C:91",
        "HA04_P":      "C:95",
        "HA04_N":      "C:97",
        "HA02_P":      "C:99",
        "HA02_N":      "C:101",
        "HA17_P":      "C:135",
        "HA17_N":      "C:137",

        "GCLK1_M2C_P": "C:4",
        "GCLK1_M2C_N": "C:6",
        "DP4_M2C_P":   "C:16",
        "DP4_M2C_N":   "C:20",
        "DP5_M2C_P":   "C:24",
        "DP5_M2C_N":   "C:28",
        "DP6_M2C_P":   "C:32",
        "DP6_M2C_N":   "C:36",
        "DP7_M2C_P":   "C:40",
        "DP7_M2C_N":   "C:44",
        "HA16_P":      "C:48",
        "HA16_N":      "C:50",
        "HA15_P":      "C:54",
        "HA15_N":      "C:56",
        "HA14_P":      "C:60",
        "HA14_N":      "C:62",
        "HA12_P":      "C:72",
        "HA12_N":      "C:74",
        "HA10_P":      "C:78",
        "HA10_N":      "C:80",
        "HA08_P":      "C:82",
        "HA08_N":      "C:84",
        "HA00_CC_P":   "C:88",
        "HA00_CC_N":   "C:90",
        "HA06_P":      "C:92",
        "HA06_N":      "C:94",
        "HA05_P":      "C:98",
        "HA05_N":      "C:100",
        "HA03_P":      "C:102",
        "HA03_N":      "C:104",
    }),

    ("IO1", {
        "CLK_P": "C:122",
        "CLK_N": "C:124",
         "D0_P": "C:115",
         "D1_N": "C:117",
         "D2_P": "C:129",
         "D3_N": "C:131",
         "D4_P": "C:125",
         "D5_N": "C:127",
         "D6_P": "C:119",
         "D7_N": "C:121",
         "D8_P": "C:109",
         "D9_N": "C:111",
        "D10_P": "C:105",
        "D11_N": "C:107",
        "D12_P": "C:138",
        "D13_N": "C:140",
        "D14_P": "C:132",
        "D15_N": "C:134",
        "D16_P": "C:128",
        "D17_N": "C:130",
        "D18_P": "C:118",
        "D19_N": "C:120",
        "D20_P": "C:112",
        "D21_N": "C:114",
        "D22_P": "C:108",
        "D23_N": "C:110",
    }),]
